<?php
/**
 * Ajax Functions
 *
 * @package     AutomatorWP\Integrations\Twitter\Ajax_Functions
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * To send tweet
 *
 * @since 1.0.0
 *
 * @param string  $oauth_header
 * @param string  $tweet
 * 
 * @return string
 */
function automatorwp_twitter_send_tweet( $oauth_header, $tweet ) {
    
    // Endpoint to send the tweet
    $url = 'https://api.twitter.com/2/tweets';

    $data = json_encode([
        'text' => $tweet
    ]);

    // To send the tweet
    $response = wp_remote_post( $url, array(
        'headers' => array(
            'Authorization' => $oauth_header,
            'Content-Type' => 'application/json',
        ),
        'body' => $data,
    ));

    // Verify the response
    if (is_wp_error($response)) {
        $response_request = sprintf( __('Error: %s', 'automatorwp-pro'), $response->get_error_message() );
    } else {
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code == 201) {
            $response_request = __('Tweet sent successfully', 'automatorwp-pro');
        } else {
            $response_request = sprintf( __('Error: %s', 'automatorwp-pro'), $response->get_error_message() );
        }
    }

    return $response_request;
}

/**
 * To build base string for Oauth Authorization
 *
 * @since 1.0.0
 *
 * @param string    $baseURI    Endpoint
 * @param string    $method     POST
 * @param array     $params     Oauth params
 * 
 * @return string
 */

function automatorwp_twitter_build_base_string( $baseURI, $method, $params ) {

    $r = array();
    ksort( $params );
    foreach( $params as $key => $value ) {
        $r[] = "$key=" . rawurlencode( $value );
    }
    return $method . "&" . rawurlencode( $baseURI ) . '&' . rawurlencode( implode( '&', $r ) );
}

/**
 * Create oauth authorization header
 *
 * @since 1.0.0
 *
 * @param string  $consumer_key
 * @param string  $consumer_secret
 * @param string  $access_token
 * @param string  $access_token_secret
 * 
 * @return string
 */
function automatorwp_twitter_get_oauth_header( $consumer_key, $consumer_secret, $access_token, $access_token_secret ) {

    // Endpoint to send the tweet
    $url = 'https://api.twitter.com/2/tweets';

    // To create the Oauth signature
    $oauth = [
        'oauth_consumer_key' => $consumer_key,
        'oauth_token' => $access_token,
        'oauth_nonce' => (string) mt_rand(), // Una cadena única para esta solicitud
        'oauth_timestamp' => time(),
        'oauth_signature_method' => 'HMAC-SHA1',
        'oauth_version' => '1.0'
    ];

    $base_info = automatorwp_twitter_build_base_string( $url, 'POST', $oauth );
    $secret_key = rawurlencode( $consumer_secret ) . '&' . rawurlencode( $access_token_secret );
    $oauth_signature = base64_encode( hash_hmac('sha1', $base_info, $secret_key, true ) );
    $oauth['oauth_signature'] = $oauth_signature;

    // Authorization header
    $oauth_header = array();
    foreach( $oauth as $key => $value ) {
        $oauth_header[] = "$key=\"" . rawurlencode( $value ) . "\"";
    }
    $oauth_header = 'OAuth ' . implode( ', ', $oauth_header );    

    return $oauth_header;

}